﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class AssignIpv6AddressesRequest : public EC2Request {
 public:
  AWS_EC2_API AssignIpv6AddressesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssignIpv6Addresses"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The number of IPv6 prefixes that Amazon Web Services automatically assigns to
   * the network interface. You cannot use this option if you use the
   * <code>Ipv6Prefixes</code> option.</p>
   */
  inline int GetIpv6PrefixCount() const { return m_ipv6PrefixCount; }
  inline bool Ipv6PrefixCountHasBeenSet() const { return m_ipv6PrefixCountHasBeenSet; }
  inline void SetIpv6PrefixCount(int value) {
    m_ipv6PrefixCountHasBeenSet = true;
    m_ipv6PrefixCount = value;
  }
  inline AssignIpv6AddressesRequest& WithIpv6PrefixCount(int value) {
    SetIpv6PrefixCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more IPv6 prefixes assigned to the network interface. You can't use
   * this option if you use the <code>Ipv6PrefixCount</code> option.</p>
   */
  inline const Aws::Vector<Aws::String>& GetIpv6Prefixes() const { return m_ipv6Prefixes; }
  inline bool Ipv6PrefixesHasBeenSet() const { return m_ipv6PrefixesHasBeenSet; }
  template <typename Ipv6PrefixesT = Aws::Vector<Aws::String>>
  void SetIpv6Prefixes(Ipv6PrefixesT&& value) {
    m_ipv6PrefixesHasBeenSet = true;
    m_ipv6Prefixes = std::forward<Ipv6PrefixesT>(value);
  }
  template <typename Ipv6PrefixesT = Aws::Vector<Aws::String>>
  AssignIpv6AddressesRequest& WithIpv6Prefixes(Ipv6PrefixesT&& value) {
    SetIpv6Prefixes(std::forward<Ipv6PrefixesT>(value));
    return *this;
  }
  template <typename Ipv6PrefixesT = Aws::String>
  AssignIpv6AddressesRequest& AddIpv6Prefixes(Ipv6PrefixesT&& value) {
    m_ipv6PrefixesHasBeenSet = true;
    m_ipv6Prefixes.emplace_back(std::forward<Ipv6PrefixesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the network interface.</p>
   */
  inline const Aws::String& GetNetworkInterfaceId() const { return m_networkInterfaceId; }
  inline bool NetworkInterfaceIdHasBeenSet() const { return m_networkInterfaceIdHasBeenSet; }
  template <typename NetworkInterfaceIdT = Aws::String>
  void SetNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    m_networkInterfaceIdHasBeenSet = true;
    m_networkInterfaceId = std::forward<NetworkInterfaceIdT>(value);
  }
  template <typename NetworkInterfaceIdT = Aws::String>
  AssignIpv6AddressesRequest& WithNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    SetNetworkInterfaceId(std::forward<NetworkInterfaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IPv6 addresses to be assigned to the network interface. You can't use
   * this option if you're specifying a number of IPv6 addresses.</p>
   */
  inline const Aws::Vector<Aws::String>& GetIpv6Addresses() const { return m_ipv6Addresses; }
  inline bool Ipv6AddressesHasBeenSet() const { return m_ipv6AddressesHasBeenSet; }
  template <typename Ipv6AddressesT = Aws::Vector<Aws::String>>
  void SetIpv6Addresses(Ipv6AddressesT&& value) {
    m_ipv6AddressesHasBeenSet = true;
    m_ipv6Addresses = std::forward<Ipv6AddressesT>(value);
  }
  template <typename Ipv6AddressesT = Aws::Vector<Aws::String>>
  AssignIpv6AddressesRequest& WithIpv6Addresses(Ipv6AddressesT&& value) {
    SetIpv6Addresses(std::forward<Ipv6AddressesT>(value));
    return *this;
  }
  template <typename Ipv6AddressesT = Aws::String>
  AssignIpv6AddressesRequest& AddIpv6Addresses(Ipv6AddressesT&& value) {
    m_ipv6AddressesHasBeenSet = true;
    m_ipv6Addresses.emplace_back(std::forward<Ipv6AddressesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of additional IPv6 addresses to assign to the network interface.
   * The specified number of IPv6 addresses are assigned in addition to the existing
   * IPv6 addresses that are already assigned to the network interface. Amazon EC2
   * automatically selects the IPv6 addresses from the subnet range. You can't use
   * this option if specifying specific IPv6 addresses.</p>
   */
  inline int GetIpv6AddressCount() const { return m_ipv6AddressCount; }
  inline bool Ipv6AddressCountHasBeenSet() const { return m_ipv6AddressCountHasBeenSet; }
  inline void SetIpv6AddressCount(int value) {
    m_ipv6AddressCountHasBeenSet = true;
    m_ipv6AddressCount = value;
  }
  inline AssignIpv6AddressesRequest& WithIpv6AddressCount(int value) {
    SetIpv6AddressCount(value);
    return *this;
  }
  ///@}
 private:
  int m_ipv6PrefixCount{0};

  Aws::Vector<Aws::String> m_ipv6Prefixes;

  Aws::String m_networkInterfaceId;

  Aws::Vector<Aws::String> m_ipv6Addresses;

  int m_ipv6AddressCount{0};
  bool m_ipv6PrefixCountHasBeenSet = false;
  bool m_ipv6PrefixesHasBeenSet = false;
  bool m_networkInterfaceIdHasBeenSet = false;
  bool m_ipv6AddressesHasBeenSet = false;
  bool m_ipv6AddressCountHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
