﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lambda/LambdaRequest.h>
#include <aws/lambda/Lambda_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace Lambda {
namespace Model {

/**
 */
class RemovePermissionRequest : public LambdaRequest {
 public:
  AWS_LAMBDA_API RemovePermissionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RemovePermission"; }

  AWS_LAMBDA_API Aws::String SerializePayload() const override;

  AWS_LAMBDA_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name or ARN of the Lambda function, version, or alias.</p> <p
   * class="title"> <b>Name formats</b> </p> <ul> <li> <p> <b>Function name</b> –
   * <code>my-function</code> (name-only), <code>my-function:v1</code> (with
   * alias).</p> </li> <li> <p> <b>Function ARN</b> –
   * <code>arn:aws:lambda:us-west-2:123456789012:function:my-function</code>.</p>
   * </li> <li> <p> <b>Partial ARN</b> –
   * <code>123456789012:function:my-function</code>.</p> </li> </ul> <p>You can
   * append a version number or alias to any of the formats. The length constraint
   * applies only to the full ARN. If you specify only the function name, it is
   * limited to 64 characters in length.</p>
   */
  inline const Aws::String& GetFunctionName() const { return m_functionName; }
  inline bool FunctionNameHasBeenSet() const { return m_functionNameHasBeenSet; }
  template <typename FunctionNameT = Aws::String>
  void SetFunctionName(FunctionNameT&& value) {
    m_functionNameHasBeenSet = true;
    m_functionName = std::forward<FunctionNameT>(value);
  }
  template <typename FunctionNameT = Aws::String>
  RemovePermissionRequest& WithFunctionName(FunctionNameT&& value) {
    SetFunctionName(std::forward<FunctionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Statement ID of the permission to remove.</p>
   */
  inline const Aws::String& GetStatementId() const { return m_statementId; }
  inline bool StatementIdHasBeenSet() const { return m_statementIdHasBeenSet; }
  template <typename StatementIdT = Aws::String>
  void SetStatementId(StatementIdT&& value) {
    m_statementIdHasBeenSet = true;
    m_statementId = std::forward<StatementIdT>(value);
  }
  template <typename StatementIdT = Aws::String>
  RemovePermissionRequest& WithStatementId(StatementIdT&& value) {
    SetStatementId(std::forward<StatementIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify a version or alias to remove permissions from a published version of
   * the function.</p>
   */
  inline const Aws::String& GetQualifier() const { return m_qualifier; }
  inline bool QualifierHasBeenSet() const { return m_qualifierHasBeenSet; }
  template <typename QualifierT = Aws::String>
  void SetQualifier(QualifierT&& value) {
    m_qualifierHasBeenSet = true;
    m_qualifier = std::forward<QualifierT>(value);
  }
  template <typename QualifierT = Aws::String>
  RemovePermissionRequest& WithQualifier(QualifierT&& value) {
    SetQualifier(std::forward<QualifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update the policy only if the revision ID matches the ID that's specified.
   * Use this option to avoid modifying a policy that has changed since you last read
   * it.</p>
   */
  inline const Aws::String& GetRevisionId() const { return m_revisionId; }
  inline bool RevisionIdHasBeenSet() const { return m_revisionIdHasBeenSet; }
  template <typename RevisionIdT = Aws::String>
  void SetRevisionId(RevisionIdT&& value) {
    m_revisionIdHasBeenSet = true;
    m_revisionId = std::forward<RevisionIdT>(value);
  }
  template <typename RevisionIdT = Aws::String>
  RemovePermissionRequest& WithRevisionId(RevisionIdT&& value) {
    SetRevisionId(std::forward<RevisionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_functionName;

  Aws::String m_statementId;

  Aws::String m_qualifier;

  Aws::String m_revisionId;
  bool m_functionNameHasBeenSet = false;
  bool m_statementIdHasBeenSet = false;
  bool m_qualifierHasBeenSet = false;
  bool m_revisionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lambda
}  // namespace Aws
